/**
 * StockOne 크롬 익스텐션 컨텐츠 스크립트
 * 토스증권 페이지 DOM에서 포트폴리오 데이터 추출
 */

console.log('[Content Script] StockOne loaded on:', window.location.href);

/**
 * 백그라운드로부터 메시지 수신
 */
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  console.log('[Content Script] Received message:', message.type);

  if (message.type === 'PARSE_PORTFOLIO') {
    parsePortfolio()
      .then((data) => {
        sendResponse({ success: true, data });
      })
      .catch((error) => {
        console.error('[Content Script] Parse error:', error);
        sendResponse({ success: false, error: error.message });
      });

    return true; // 비동기 응답 대기
  }
});

/**
 * 토스증권 페이지에서 포트폴리오 데이터 파싱
 * 페이지 전체 텍스트를 추출하여 반환 (서버에서 AI 파싱)
 * @returns {Promise<Object>} 파싱된 포트폴리오 데이터
 */
async function parsePortfolio() {
  console.log('[Content Script] Parsing portfolio...');

  // 페이지가 완전히 로드될 때까지 잠시 대기
  await new Promise(resolve => setTimeout(resolve, 1000));

  try {
    // 방법 1: 페이지 전체 텍스트 추출 (서버에서 AI 파싱)
    const pageText = document.body.innerText;

    console.log('[Content Script] Page text length:', pageText.length);

    // 텍스트가 너무 짧으면 페이지가 아직 로딩 중
    if (pageText.length < 100) {
      throw new Error('페이지가 아직 로딩 중입니다. 잠시 후 다시 시도하세요.');
    }

    return {
      rawText: pageText,
      holdings: [],
      summary: {},
      timestamp: new Date().toISOString(),
      source: 'tossinvest',
      url: window.location.href,
    };
  } catch (error) {
    console.error('[Content Script] Parse error:', error);
    throw error;
  }
}

/**
 * 대체 DOM 구조 파싱 (토스증권 UI 업데이트 대응)
 */
function parseAlternativeStructure(items) {
  const holdings = [];

  items.forEach((item) => {
    // 텍스트 전체를 가져와서 파싱
    const text = item.innerText || item.textContent;
    const lines = text.split('\n').map((l) => l.trim()).filter((l) => l);

    if (lines.length >= 3) {
      holdings.push({
        name: lines[0], // 첫 번째 라인: 종목명
        currentPrice: parseNumberFromText(lines[1]), // 두 번째 라인: 현재가
        profitRate: parseNumberFromText(lines[2]), // 세 번째 라인: 수익률
      });
    }
  });

  return {
    holdings,
    summary: {},
    timestamp: new Date().toISOString(),
    source: 'tossinvest',
  };
}

/**
 * 텍스트 추출 헬퍼 (여러 셀렉터 시도)
 */
function extractText(parent, ...selectors) {
  for (const selector of selectors) {
    const el = parent.querySelector(selector);
    if (el) {
      return el.innerText?.trim() || el.textContent?.trim() || '';
    }
  }
  return '';
}

/**
 * 숫자 추출 헬퍼 (쉼표, % 제거 후 변환)
 */
function extractNumber(parent, ...selectors) {
  const text = extractText(parent, ...selectors);
  return parseNumberFromText(text);
}

/**
 * 텍스트에서 숫자 파싱
 */
function parseNumberFromText(text) {
  if (!text) return null;

  // 쉼표, 공백, % 제거
  const cleaned = text.replace(/[,\s%]/g, '');

  // +/- 기호 처리
  const number = parseFloat(cleaned);

  return isNaN(number) ? null : number;
}

/**
 * 특정 요소가 나타날 때까지 대기
 * @param {string} selector - CSS 셀렉터
 * @param {number} timeout - 타임아웃 (ms)
 * @returns {Promise<Element>}
 */
function waitForElement(selector, timeout = 3000) {
  return new Promise((resolve, reject) => {
    const el = document.querySelector(selector);
    if (el) {
      resolve(el);
      return;
    }

    const observer = new MutationObserver((mutations, obs) => {
      const el = document.querySelector(selector);
      if (el) {
        obs.disconnect();
        resolve(el);
      }
    });

    observer.observe(document.body, {
      childList: true,
      subtree: true,
    });

    setTimeout(() => {
      observer.disconnect();
      reject(new Error(`Timeout waiting for selector: ${selector}`));
    }, timeout);
  });
}

/**
 * 페이지 로드 완료 시 자동 알림 (선택적)
 */
window.addEventListener('load', () => {
  console.log('[Content Script] Page loaded');

  // 백그라운드에 페이지 로드 완료 알림
  chrome.runtime.sendMessage({
    type: 'PAGE_LOADED',
    url: window.location.href,
  }).catch(() => {
    // 백그라운드가 응답하지 않으면 무시
  });
});
