/**
 * StockOne 크롬 익스텐션 백그라운드 서비스 워커
 * - 사이드 패널 제어
 * - 메시지 라우팅 (sidepanel ↔ content script)
 * - API 호출 프록시 (CORS 우회)
 */

const SERVER_URL = 'https://stockone.heywook.com';

/**
 * 익스텐션 설치 또는 업데이트 시 실행
 */
chrome.runtime.onInstalled.addListener(() => {
  console.log('StockOne Extension installed');
});

/**
 * 익스텐션 아이콘 클릭 시 사이드 패널 열기
 */
chrome.action.onClicked.addListener(async (tab) => {
  // 현재 탭에서 사이드 패널 열기
  await chrome.sidePanel.open({ tabId: tab.id });
});

/**
 * 메시지 라우팅 핸들러
 * sidepanel과 content script 간의 메시지를 중계
 */
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  console.log('[Background] Received message:', message.type, message);

  switch (message.type) {
    case 'PARSE_PORTFOLIO':
      // 사이드 패널 → Content Script: 포트폴리오 파싱 요청
      handleParsePortfolio(sender, sendResponse);
      return true; // 비동기 응답 대기

    case 'SAVE_PORTFOLIO':
      // 사이드 패널 → 백그라운드: 포트폴리오 저장 요청
      handleSavePortfolio(message.rawText, sendResponse);
      return true;

    case 'PORTFOLIO_DATA':
      // Content Script → 사이드 패널: 파싱된 데이터 전달
      handlePortfolioData(message.data, sendResponse);
      return true;

    case 'API_REQUEST':
      // 사이드 패널 → 백그라운드: API 호출 프록시
      handleApiRequest(message, sendResponse);
      return true;

    default:
      console.warn('[Background] Unknown message type:', message.type);
  }
});

/**
 * 포트폴리오 파싱 요청 처리
 * 현재 활성 탭에 메시지 전송 후 서버 API로 텍스트 파싱 요청
 */
async function handleParsePortfolio(sender, sendResponse) {
  try {
    // 현재 활성 탭 가져오기
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });

    if (!tab) {
      sendResponse({ success: false, error: '활성 탭을 찾을 수 없습니다.' });
      return;
    }

    // 토스증권 페이지인지 확인
    if (!tab.url?.includes('tossinvest.com')) {
      sendResponse({
        success: false,
        error: '토스증권 페이지에서만 포트폴리오를 수집할 수 있습니다.',
      });
      return;
    }

    // Content Script에 파싱 요청
    chrome.tabs.sendMessage(
      tab.id,
      { type: 'PARSE_PORTFOLIO' },
      async (response) => {
        if (chrome.runtime.lastError) {
          const errorMsg = chrome.runtime.lastError.message || '알 수 없는 오류';
          console.error('[Background] Content script error:', errorMsg);
          sendResponse({
            success: false,
            error: `Content Script 오류: ${errorMsg}. 토스증권 페이지에서 새로고침 후 다시 시도하세요.`,
          });
          return;
        }

        // Content script 응답 확인
        if (!response || !response.success) {
          sendResponse({
            success: false,
            error: response?.error || '페이지 데이터를 가져올 수 없습니다.',
          });
          return;
        }

        // rawText가 있으면 서버 API로 파싱 요청
        if (response.data?.rawText) {
          try {
            console.log('[Background] Sending text to server for parsing...');

            // 저장된 토큰 가져오기
            const storage = await chrome.storage.local.get('authToken');
            const token = storage.authToken;

            const apiResponse = await fetch(`${SERVER_URL}/api/portfolio/text`, {
              method: 'POST',
              headers: {
                'Content-Type': 'application/json',
                ...(token ? { 'Authorization': `Bearer ${token}` } : {}),
              },
              body: JSON.stringify({
                text: response.data.rawText,
                source: 'tossinvest-extension',
                preview: true, // 미리보기 모드 - 저장하지 않음
              }),
            });

            const apiData = await apiResponse.json();

            if (apiData.success && apiData.data) {
              console.log('[Background] Server parsed portfolio:', apiData.data);
              // API 응답을 PortfolioCard가 기대하는 형식으로 변환
              sendResponse({
                success: true,
                preview: apiData.preview || false,
                rawText: response.data.rawText, // 저장 시 사용할 원본 텍스트
                data: {
                  holdings: (apiData.data.holdings || []).map(h => ({
                    name: h.stockName,
                    code: h.ticker,
                    quantity: h.quantity,
                    avgPrice: h.avgPrice,
                    currentPrice: h.currentPrice,
                    valuation: h.evalAmount,
                    profitRate: h.profitRate,
                    profitAmount: h.evalAmount - (h.avgPrice * h.quantity),
                  })),
                  summary: {
                    totalValuation: apiData.data.totalAmount || 0,
                    totalProfit: apiData.data.totalProfit || 0,
                    totalProfitRate: apiData.data.totalProfit || 0,
                  },
                  timestamp: new Date().toISOString(),
                  source: 'tossinvest',
                },
              });
            } else {
              sendResponse({
                success: false,
                error: apiData.message || '서버에서 포트폴리오를 파싱할 수 없습니다.',
              });
            }
          } catch (apiError) {
            console.error('[Background] API error:', apiError);
            sendResponse({
              success: false,
              error: `서버 연결 오류: ${apiError.message}`,
            });
          }
        } else {
          // rawText가 없으면 그대로 응답 (기존 방식)
          sendResponse(response);
        }
      }
    );
  } catch (error) {
    console.error('[Background] Parse portfolio error:', error);
    sendResponse({ success: false, error: error.message });
  }
}

/**
 * 포트폴리오 저장 요청 처리
 * preview로 파싱된 데이터를 실제로 저장
 */
async function handleSavePortfolio(rawText, sendResponse) {
  try {
    console.log('[Background] Saving portfolio...');

    // 저장된 토큰 가져오기
    const storage = await chrome.storage.local.get('authToken');
    const token = storage.authToken;

    const apiResponse = await fetch(`${SERVER_URL}/api/portfolio/text`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        ...(token ? { 'Authorization': `Bearer ${token}` } : {}),
      },
      body: JSON.stringify({
        text: rawText,
        source: 'tossinvest-extension',
        preview: false, // 실제 저장
      }),
    });

    const apiData = await apiResponse.json();

    if (apiData.success) {
      console.log('[Background] Portfolio saved:', apiData.data?.id);
      sendResponse({
        success: true,
        message: apiData.message,
        data: apiData.data,
      });
    } else {
      sendResponse({
        success: false,
        error: apiData.message || '포트폴리오 저장에 실패했습니다.',
      });
    }
  } catch (error) {
    console.error('[Background] Save portfolio error:', error);
    sendResponse({
      success: false,
      error: error.message,
    });
  }
}

/**
 * 파싱된 포트폴리오 데이터 처리
 * (필요 시 추가 가공 또는 저장)
 */
function handlePortfolioData(data, sendResponse) {
  console.log('[Background] Portfolio data received:', data);

  // 사이드 패널로 데이터 전달 (브로드캐스트)
  chrome.runtime.sendMessage({
    type: 'PORTFOLIO_DATA',
    data: data,
  });

  sendResponse({ success: true });
}

/**
 * API 호출 프록시 (CORS 우회)
 * 사이드 패널에서 직접 API 호출 시 CORS 문제를 해결
 */
async function handleApiRequest(message, sendResponse) {
  const { method, endpoint, body, headers } = message;

  try {
    const url = `${SERVER_URL}${endpoint}`;
    const fetchOptions = {
      method: method || 'GET',
      headers: {
        'Content-Type': 'application/json',
        ...headers,
      },
    };

    if (body) {
      fetchOptions.body = JSON.stringify(body);
    }

    const response = await fetch(url, fetchOptions);
    const data = await response.json();

    sendResponse({
      success: response.ok,
      status: response.status,
      data: data,
    });
  } catch (error) {
    console.error('[Background] API request error:', error);
    sendResponse({
      success: false,
      error: error.message,
    });
  }
}

/**
 * 탭 업데이트 감지 (토스증권 페이지 접속 시)
 */
chrome.tabs.onUpdated.addListener((tabId, changeInfo, tab) => {
  // 페이지 로딩 완료 & 토스증권 페이지
  if (changeInfo.status === 'complete' && tab.url?.includes('tossinvest.com')) {
    console.log('[Background] 토스증권 페이지 감지:', tab.url);

    // 사이드 패널에 알림 (연결된 경우만)
    chrome.runtime.sendMessage({
      type: 'PAGE_READY',
      url: tab.url,
    }).catch(() => {
      // 사이드 패널이 열려있지 않으면 에러 무시
    });
  }

  // URL 변경 감지 (종목 페이지 감지)
  if (changeInfo.url && tab.url) {
    // URL 패턴: https://tossinvest.com/stocks/A005930 또는 /stocks/005930
    const match = tab.url.match(/tossinvest\.com\/stocks\/A?(\d{6})/);
    if (match) {
      const stockCode = match[1];
      console.log('[Background] 종목 감지:', stockCode);

      // 사이드 패널에 메시지 전송
      chrome.runtime.sendMessage({
        type: 'STOCK_DETECTED',
        stockCode: stockCode,
      }).catch(() => {
        // 사이드 패널이 열려있지 않으면 무시
      });
    }
  }
});
